#!/usr/bin/perl

# delete files more slowly than 'rm'
# change delay between deletions by sending signal USR1 and USR2
# --Andrew Chasse <prog@albedo.ca> (20090418)

use strict;
use Time::HiRes qw(usleep gettimeofday);

$SIG{'USR1'} = \&increase_delay;
$SIG{'USR2'} = \&decrease_delay;

my $delay = 0;
my $debug = 1;
my $start_time = gettimeofday();
my $c_deleted = 0;
my $inspect_every_x = 100;
my $stats_every_x = 500;

my $what = $ARGV[0] || die "Usage: $0 <dir> [<delay in micro-sec between deletes>] [debug verbosity: 0 | 1 | 2]\nSend signal USR1 to increase delay, USR2 to decrease delay\n";
die "dir [$what] is not a directory" if(! -d $what);

my $delay = $ARGV[1];
$delay = 10000 if(!length($delay));
&logger(1,"using delay of $delay microseconds");

$debug = $ARGV[2] || 1;
&logger(1,"using debug verbosity of $debug");

delete_recursive($what);
&stats(1);

sub delete_recursive {
  my ($dir) = @_;
  my $dh;   # directory handle

  &logger(1,"delete_recursive: beginning recursive delete of [$dir]");

  if(!opendir($dh,$dir)) {
    &logger(0,"delete_recursive: failed to opendir [$dir], skipping: $!");
    return 0;
  }

  while(my $entry = readdir($dh)) {
    next if($entry =~ /^\.\.{0,1}$/);  # skip ./ and ../
    my $file = "$dir/$entry";
    if(-d $file) {
      delete_recursive($file);
    }
    else {
      if(!unlink($file)) {
        &logger(0,"delete_recursive: failed to unlink [$file], skipping: $!");
        next;
      }
      &logger(2,"$file removed");
    }

    $c_deleted++;
    if(($c_deleted % $inspect_every_x) == 0) {
      &analyze_system_load;
    }
    if(($c_deleted % $stats_every_x) == 0) {
      &stats(0);
    }
    usleep($delay);
  }

  closedir($dh);
  if(!rmdir($dir)) {
    &logger(0,"delete_recursive: failed to rmdir [$dir]: $!");
    return 0;
  }
  &logger(1,"$dir/ rmdir'd");
  return 1;
}

sub analyze_system_load {
  &logger(2,"analyze_system_load: would have checked system load here and done something with delay $delay");
}

sub stats {
  my ($force) = @_;
  if($force || ($c_deleted % $stats_every_x) == 0) {
    my $delta_time = (gettimeofday() - $start_time) || 0.1;
    my $delta_time_human = int($delta_time * 100) / 100;
    my $deleted_per_sec = int($c_deleted / $delta_time * 100) / 100;
    my $deleted_per_hour = $deleted_per_sec * 3600;
    &logger(1,"stats: deleted [$c_deleted] in ${delta_time_human}s, $deleted_per_sec deleted/s, $deleted_per_hour deleted/h");
  }
}

sub logger {
  my($level,$msg) = @_;

  if($level <= $debug) {
    print gmtime() . " $msg\n";
  }
}

sub increase_delay {
  $delay += 10000;
  &logger(0,"delay increased to $delay microseconds");
}

sub decrease_delay {
  $delay -= 10000;
  $delay = 0 if($delay < 0);
  &logger(0,"delay decreased to $delay microseconds");
}
